/**
  * TimeConverter.java
  *
  * Version:
  *	$Id$
  *
  * Revisions:
  *     $Log$
  */

import java.awt.*;
import java.awt.event.*;

/**
 * This program will convert time entered in 10ths of a
 * second to HH:MM:SS.T form.  It uses the TimeCounter
 * class to handle the conversion and builds a GUI to
 * communicate with the user.
 */

public class TimeConverter implements ActionListener, TextListener {
    private Frame win;                    // Frame for the GUI
    private TimeCounter theTime;          // The TimeCounter for conversions
    private IntegerTextField rawTime;     // Input field for the raw time
    private Label processedTime;          // A place to display formatted time

    /**
     * Create a time converter with a specified title
     *
     * @param    String    Title for the TimeConverter window
     */

    public TimeConverter( String id ) {
        win = new Frame( id );

        // Frames do not handle close by default ...

        win.addWindowListener(
                new WindowAdapter() {
                    public void windowClosing( WindowEvent e ) {
                        System.exit( 0 );
                    }
                }
        );

        // Do the Buttons
  
        Panel buttonPanel = new Panel();

        Button b = new Button( "Inc" );
        b.addActionListener( this );
        buttonPanel.add( b );

        b = new Button( "Reset" );
        b.addActionListener( this );
        buttonPanel.add( b );
  
        b = new Button( "Quit" );
        b.addActionListener( this );
        buttonPanel.add( b);

        win.add( "South", buttonPanel );
    
        // Create the main "input area" using a GridBag layout manager

        Panel mainPanel = new Panel();

        mainPanel.setLayout( new GridBagLayout() );
        GridBagConstraints c = new GridBagConstraints();
        c.anchor = GridBagConstraints.WEST;
        c.insets = new Insets( 2, 2, 2, 2 );
        c.weightx = c.weighty = 1.0;
  
        c.gridx=0; c.gridy=0;
        mainPanel.add( new Label( "Elapsed time in 10ths of sec:" ), c );

        c.gridx=0; c.gridy=1;
        mainPanel.add( new Label( "Elapsed time in HH:MM:SS.T:" ), c );

        c = new GridBagConstraints();
        c.anchor = GridBagConstraints.WEST;
        c.fill = GridBagConstraints.HORIZONTAL;
        c.gridwidth=GridBagConstraints.REMAINDER;
        c.insets = new Insets( 2, 2, 2, 2 );
        c.weightx = c.weighty = 1.0;

        c.gridx = 1; 
	c.gridy = 0;
        rawTime = new IntegerTextField( 10 );
        mainPanel.add( rawTime, c );
        rawTime.addTextListener( this );

        theTime = new TimeCounter();

        c.gridx = 1; 
        c.gridy = 1;
        processedTime = new Label();
        mainPanel.add( processedTime, c );

        win.add( "Center", mainPanel );
    
        // Initialize the formatted time display and prepare GUI for display

        updateDisplay();

        win.pack();

        // Center the window on the screen

        Dimension screenSize = Toolkit.getDefaultToolkit().getScreenSize();
        Dimension mySize = win.getSize();

        int x = ( screenSize.width/2 ) - ( mySize.width/2 );
        int y = ( screenSize.height/2 ) - ( mySize.height/2 );

        win.setBounds( x, y, mySize.width, mySize.height );

        // Let it run!!

        win.show();
    }

    /**
     * Take the current time from the TimeCounter and display
     * it in HH:MM:SS.T format.  Note that the real conversion
     * work is handled by the TimeCounter.
     */

    private void updateDisplay() {

        // Display leading 0s by appending a 0 to the front of the
        // string and then taking the two rightmost characters from
        // the resulting string (i.e. 010 becomes 10 and 02 stays 02).

        String minutes = "0"+theTime.getMinutes();
        minutes = minutes.substring( minutes.length() - 2 );

        String seconds = "0"+theTime.getSeconds();
        seconds = seconds.substring( seconds.length() - 2 );

        // Build the formatted string and display the result in the label

        processedTime.setText( theTime.getHours() + ":" 
                               + minutes + ":"
                               + seconds + "."
                               + theTime.getTSeconds() );
    }

    /**
     * Handle button events
     */

    public void actionPerformed( ActionEvent e ) {
        String cmd = e.getActionCommand();

        // Quit simply terminates the program.
        // Inc will add one tot he time and update the display
        // Reset sets the time to zero and updates the display

        if ( cmd.equals( "Quit") ) {
            win.dispose();
            System.exit( 0 );
        } else if ( cmd.equals( "Inc" ) ) {
            theTime.tick();
            rawTime.setText( Integer.toString( theTime.getTime() ) );
            updateDisplay();
        } else if ( cmd.equals( "Reset" ) ) {
            theTime.reset();
            rawTime.setText( "" );
            updateDisplay();
        }
    }

    /**
     * Whenever the time in the input box changes, change
     * the time stored in the TimeCounter and update the
     * display.
     */

    public void textValueChanged( TextEvent ev ) {
        int curVal=0;
    
        // An exception should never occur here since InputTextField
        // filters characters and makes sure that only digits
        // are entered.

        try {
            curVal=Integer.parseInt( rawTime.getText() );
        }
        catch( Exception e ) {};

        theTime.setTime( curVal );
        updateDisplay();
    }

    /**
     * A simple main to start up and display a TimeConverter.
     *
     * @param args command line arguments - ignored
     *
     */
  
    public static void main( String args[] ) {
        TimeConverter gui = new TimeConverter( "Time Converter" );
    }

} // TimeConverter
