/**
 * TimerLabel.java
 *
 * Version:
 *	$Id$
 *
 * Revsions:
 *     $Log$
 */

import java.awt.*;
import java.awt.event.*;

/**
 * This label displays a time in HH:MM:SS.T format.  It
 * runs as a thread so that the time can be updated in
 * real time (or at least as real as Java gets).
 */

public class TimerLabel extends Label implements Runnable {
    private TimeCounter time;     // The keeper of the time

    String nums[];                // To make conversion from int to
                                  // string fast, this array will
                                  // hold string representation of the
                                  // numbers from 0 to 99

    private boolean running;      // Are we running (i.e. keeping time)?
    private boolean update;       // Should the display be updated?

    Thread updateThread;          // The time thread

    /**
     * Create a TimerLabel that is initialized to 0
     */

    public TimerLabel() {
        this( 0 );
    }

    /**
     * Create a TimerLabel initialized to the given number
     * of seconds.
     *
     * @param    int    Initial number of seconds to display in the label
     */
  
    public TimerLabel( int secs ) {
        // Initialize the label

        super( "00:00:00.0", Label.CENTER );

        // Put the string representations of the numbers 0 through 99
        // in the array nums.  This make conversion of integer values
        // to string pretty fast (simple table lookup).

        nums = new String[100];
        for ( int i = 0; i < 100; i++ ) {
            nums[i] = Integer.toString( i );
            if ( i < 10 ) {
	        nums[i] = "0" + nums[i];
	    }
        }

        // Use a Timecounter to keep track of the time

        time = new TimeCounter( secs );
        updateText();

        // Start the update thread

        updateThread = new Thread( this );
        updateThread.start();
    }

    /**
     * run() method for the thread.  Simply sleeps for a 10th
     * of a second and then increments the time and updates the
     * display.  Note that this stop watch will not be 100%
     * accurate since the time it takes to update the display is
     * note taken into account.
     */

    public void run() {
        while ( true ) {
            try {
                Thread.currentThread().sleep( 100 );
            }
            catch ( InterruptedException e ) {}

            if ( running ) {
                time.tick();
                if ( update ) {
		    updateText();
		}
            }              
        }
    }

    /**
     * Reset and stop the timer.
     */
  
    public void timerReset() {
        time.reset();
        updateText();

        running = false;
        update = true;
    }

    /**
     * Start the timer.
     */

    public void timerStart() {
        running = true;
        update = true;
    }

    /**
     * Stop the timer.
     */

    public void timerStop() {
        running = false;
        update = true;
    }

    /**
     * Freeze the display but continue counting time (i.e.
     * equivalent to lap mode on a stop watch).
     */

    public void timerNoUpdate() {
        update = false;
    }
  
    /**
     * Update the time that is being displayed.
     */

    public void updateText() {
        setText( nums[time.getHours() % 100] + ":" 
                + nums[time.getMinutes()] + ":" 
                + nums[time.getSeconds()] + "." 
                + time.getTSeconds());
    }

    /**
     * Create a timer and display it.  Initial time can
     * be set via the command line.
     *
     * @param args command line arguments - initial time
     *
     */
  
    public static void main( String args[]) {
        Frame win = new Frame( "Timer" );

        // Frames do not handle close by default ...

        win.addWindowListener(
                new WindowAdapter() {
                    public void windowClosing( WindowEvent e ) {
                        System.exit( 0 );
                    }
                }
        );

        int startTime = 0;

        // See if there is a parseable time on the command line

        try {
            if ( args.length != 0 ) {
		startTime = Integer.parseInt( args[0] );
	    }
        }
        catch ( Exception e ) {}

        // Create the label and display it in the frame.

        TimerLabel l = new TimerLabel( startTime );

        win.add( "Center", l );

        win.pack();
        win.show();

        // Make time march on
  
        l.timerStart();
    }

} // TimerLabel
